/* 
** OPTMUM.ARC - General Nonlinear Optimization
** (C) Copyright 1988, 1989 by Aptech Systems, Inc.
** All Rights Reserved.
**
**   Written in part by Ronald Schoenberg
**
**        CONTENTS                       LINE
**        --------                       ----
**        PROC OPTMUM                      19
**        Global Variables                 40
**        Source Code                     168
**
**-------------------**------------------**-------------------**-----------**
**-------------------**------------------**-------------------**-----------**
**
**   PROC OPTMUM
**
**   FORMAT                        
**          { x,f,g,cov } = optmum(x0,&fct)     
**
**   INPUT
**          x0 - vector of start values 
**   
**        &fct - pointer to a procedure that computes the function to
**               be minimized.  This procedure must have one input
**               argument, a vector of parameter values, and one 
**               output argument, the value of the function evaluated
**               at the input vector of parameter values.
**
**   OUTPUT
**          x - vector of estimated parameters
**          f - function at minimum (minus the mean log-likelihood)
**          g - gradient evaluated at x
**        cov - covariance matrix of the parameters (see __covp below)
**
**
**
**
**-------------------**------------------**-------------------**-----------**
**-------------------**------------------**-------------------**-----------**
**
**
**   GLOBAL VARIABLES                                               LINE
**
**  __title  -  title                                                 65
**  __algr   -  optimization algorithm                                68
**  __covp   -  selects covariance matrix of parameters               76
**  __step   -  selects type of step length                           85
**  __hess0  -  selects starting hessian                              96
**  __mxbkst -  number of backsteps in computing steplength          101
**  __btol   -  convergence tolerance for coefficients               105
**  __gtol   -  convergence tolerance for gradient                   109
**  __gdproc -  procedure to compute gradient                        113
**  __hsproc -  procedure to compute hessian                         134
**  __grdmd  -  numerical gradient method                            150
**  __parnms -  parameter names                                      156
**  __design -  combination of algorithms                            158
**  __diter  -  # of iters to switch algorithms for __design         170 
**  __dfct   -  criterion for change in function for __design        174
**  __miter  -  maximum number of iterations                         178
**  __dh     -  increment size for computing gradient                   
**
**
**   __title -  title of run
**
**
**    __algr -  indicator for optimization method:
**                = 0   SD (steepest descent - default)
**                = 1   BFGS (Broyden, Fletcher, Goldfarb, Shanno)
**                = 2   DFP (Davidon, Fletcher, Powell)
**                = 3   NEWTON (Newton-Raphson) 
**
**
** 
**    __covp - indicator for method of computing the covariance matrix of
**             the parameters that is returned by OPTMUM.  
**
**           = 0  the inverse of the final information matrix from 
**                       the optimization is returned in cov. (default)
**               = 1  the inverse of the second derivatives is returned.
**
**
**
**    __step - indicator determining the method for computing step length.
**              = 1,  steplength = 1
**              = 2,  cubic or quadratic steplength  (default)
**              = 3,  golden steplength.
**
**             Usually __step = 2 will be best.  If the optimization bogs
**             down try setting __step = 1 or 3.  __step = 3 will generate
**             slow iterations but faster convergence and __step = 1 will
**             generate fast iterations but slower convergence.
**            
**
**   __hess0 - determines the starting hessian for BFGS, DFP, SCORE, and
**             Newton.
**              = 0,  start with identity matrix  (default)
**              = 1,  computing starting hessian
**              or set to matrix containing user-defined starting hessian.
**
** __maxbkst - maximum number of backsteps taken to find step length.
**             Default = 10.
**
**
**    __btol - convergence tolerance for estimated coefficients.
**             Default = 1e-4.  This criterion is tested first before
**             checking the convergence tolerance for the gradient.
**
**    __gtol - convergence tolerance for gradient of estimated coefficients.
**             Default = 1e-5.  When this criterion has been satisifed
**             OPTMUM will exit the iterations.
**
**  __gdproc - pointer to a procedure that computes the gradient of the
**             function with respect to the parameters.  For example,
**             the instruction: 
**
**                     __gdproc=&gradproc 
**
**             will tell OPTMUM that a gradient procedure exists as well
**             where to find it.  The user-provided procedure has a 
**             single input argument, a vector of parameter values, and
**             a single output argument, a vector of gradients of the
**             function with respect to the parameters evaluated at the
**             vector of parameter values.  For example, suppose the
**             procedure is named gradproc and the function is a quadratic
**             function with one parameter: y=x^2+2*x+1, then
**
**                    proc gradproc(x);  retp(2*x+2); endp;
**    
**             Default = 0, i.e., no gradient procedure has been provided.
**
**
**
**  __hsproc - pointer to a procedure that computes the hessian, i.e., the
**             matrix of second order partial derivatives of the function
**             with respect to the parameters.  For example, the 
**             instruction:
**
**                    __hsproc=&hessproc;
**
**             will tell OPTMUM that a procedure has been provided for the
**             computation of the hessian and where to find it.  The
**             procedure that is provided by the user must have a single
**             input argument, the vector of parameter values, and a single 
**             output argument, the symmetric matrix of second order
**             derivatives of the function evaluated at the parameter
**             values.         
**         
**
**   __grdmd - method for computing numerical gradient. 
**                = 1, forward difference (default)
**                = 0, central difference  
**
**     __row - size of increment in computing gradient
**   
**  __parnms - character vector of parameter labels.
**
**  __design - flag for selection of an algorithmic strategy.  If __design
**             is either 1 or 2 OPTMUM will begin the iterations with
**             STEEPEST DESCENT and step length set to unity.  When the
**             function fails to improve by .001, or when 20 iterations
**             have gone by, OPTMUM will switch to the BFGS algorithm
**             and the quadratic/cubic step method.  If __design = 1
**             the Hessian will be computed at that point, or if
**             __design = 2 the BFGS iterations will begin with the 
**             Hessian set to the identity matrix.  The criterion for
**             switching algorithms can be changed by modifying the
**             global variables __diter and __dfct.
**
**   __dfct - criterion for change in function which will cause OPTMUM
**            to switch algorithms when __design is nonzero.  
**            Default = .001.
**
**  __diter - criterion for maximum number of iterations before
**            switching algorithms when __design is nonzero.
**            Default = 20.
**
**  __miter - maximum number of iterations.
*/

/*-------------------**------------------**-------------------**-----------**
**-------------------**------------------**-------------------**-----------*/

@  SOURCE CODE  @

@  internally created global variables @

declare __ml_scr ?= 0;
declare __ml_hhs ?= 0;

external matrix _grad_dh;

#include optmum.ext;
#include dtran.ext;
#include stat.ext;

PROC (4) = optmum(x0,&_mfct);

@ ------- LOCALS ----------- @
    local x,g,s,h,gflag,iter,ky,kz,old,vof,x0,d,dfct,f0,parnms,bksteps,dg,
        dx,relb,smallval,relgrad,algrm,stepm,pg,k0,k1,k2,lr,lf,ll,g0,k, 
        tstart,oldt,maxl,w0,w1,w2,w3,w4,w5,w6,w7,w8,dsgn,k4,olddsgn;
    clear bksteps,s,dfct,f0,h,dsgn;
    local _mfct:proc;

    dx = 1;
    iter = 1;
    pg = 1;
    smallval = 1e-15;
    if __output == 1;
        call csrtype(0);
    endif;
    let w1 = 2 1 2 80 0 7;
    let w2 = 5 1 24 80 0 7;
    let w3 = 1 1 1 80 0 112;
    let w4 = 3 1 4 80 0 112;
    let w5 = 6 1 24 80 0 7;
    w0 = 196*ones(1,24);
    w6 = chrs(218~w0~194~w0~194~w0~191);
    w7 = chrs(179~zeros(1,24)~179~zeros(1,24)~179~zeros(1,24)~179);
    w8 = chrs(192~w0~193~w0~193~w0~217);
    w0 = "" $+ chrs(32*ones(40-floor(strlen(__title)/2),1)) $+ __title;

    algrm = "STEEP"|"BFGS"|"DFP"|"NEWTON"|"BHHH";
    stepm = "1.0"|"STEPBT"|"GOLDEN";
    old = ndpcntrl(0,0);
    call ndpcntrl(1,1);
    if __output == 1;
        cls;
    endif;
    format /rd 10,5;

    if __design;
        __algr = 0;
        __step = 1;
        dsgn = 1;
    endif;

    maxl = typecv("maxlik");
    if not scalmiss(maxl);
        maxl = maxl eq 8;
    else;
        maxl = 0;
    endif;
    if __step < 1 or __step > 3;
        __step = 2;
    endif;
    __ml_hhs = __hsproc;
    if not maxl;
        __algr = minc(__algr|3);
    else;
        if __algr == 4;
            __hsproc = __ml_scr;
        endif;
    endif;
    if __parnms /= 0 and rows(__parnms) /= rows(x0);
        locate 2,1;
        errorlog "vector of parameter labels not equal to vector of startin"\
            "g values";
        parnms = 0;
    else;
        parnms = __parnms;
    endif;

@**************************************************************************@
@                     BEGIN OPTIMIZATION                                   @
@**************************************************************************@
    tstart = hsec;
    if __output == 1;
        gosub BAR;
    endif;

    vof = _mfct(x0);        /* Initial function value */

    if scalmiss(vof);
        if trapchk(0);
            locate 2,1;
            errorlog "ERROR:  function cannot be computed at initial parame"\
                "ter values";
            end;
        else;
            if __output == 1;
                cls;
            endif;
            retp(x0,vof,error(0),error(0));
        endif;
    endif;

    g = _deriv(x0,1,&_mfct)';       /* Initial gradient */

    relgrad = abs(g).*(maxc(abs(x0)|1)/maxc(abs(vof)|1));
    k = rows(g);    /* Number of parameters to estimate */
    if abs(g) < smallval;
        x = x0;
        goto A98;
    endif;

    if rows(__hess0) le 1;
        if __output == 1;
            scroll w1;
        endif;
        if ((__algr lt 3 or __algr eq 4) and __hess0 eq 0) or __algr eq 0;
            if __output == 1;
                locate 2,1;
                print "H set to identity matrix";
            endif;
            if __algr eq 0;
                h = 1;
            else;
                h = eye(k)*(1/maxc(abs(vof)|1));
            endif;

        elseif __algr;
            oldt = trapchk(1);
            trap 1,1;
            h = _deriv(x0,2,&_mfct);
            if not scalmiss(h);
                h = invpd(h);
            endif;
            if scalmiss(h);
                if __output == 1;
                    locate 2,1;
                    print "hessian matrix failed to invert - algorithm switc"\
                          "hed to BFGS";
                endif;
                __algr = 1;
                h = eye(k)*(1/maxc(abs(vof)|1));
            endif;
            trap oldt,1;
        endif;
    else;
        h = __hess0;
    endif;

    if k gt 48 and __output == 1;
        locate 25,15;
        printdos "\27[7m"; 
        print "  <PgDn>, <PgUp>  page parameters and gradient ";
        printdos "\27[0m";
    endif;

A0:

    /* ********* Start of iteration loop ********** */
    f0 = vof;
    if __output == 1;
        scroll w1;
        gosub BAR;
        gosub PARBOX;

        printdos "\27[7m";
        locate 3,28;
        format /ld 10,5;
        print vof;;
        locate 4,12;
        format 4,2;
        print (hsec-tstart)/100;;
        locate 3,52;
        format 6,6;
        print $algrm[__algr+1];;
        locate 3,66;
        print $stepm[__step];;
        locate 4,69;
        format 5,3;
        print s;;
        if iter > 1;
            locate 4,28;
            format /le 10,2;
            print dfct;
        endif;
        locate 4,51;
        format /ld 1,0;
        print bksteps;;
        locate 3,7;
        print iter;;
        printdos "\27[0m";

        k0 = lf;
        k2 = 0;
        do until k2 == lr;
            k2 = k2+1;
            k1 = 0;
            do until k1 == 3;
                k1 = k1+1;
                locate 7+k2,(k1-1)*25+2;
                if parnms == 0;
                    format /rd 3,0;
                    print k0;;
                    format 9,4;
                    print x0[k0];;
                    format 9,5;
                    print relgrad[k0];;
                else;
                    format /rd 8,8;
                    print $parnms[k0];;
                    format 6,2;
                    print x0[k0];;
                    format 7,4;
                    print relgrad[k0];;
                endif;
                k0 = k0+1;
                if k0 gt k;
                    goto A1;
                endif;
            endo;
        endo;
    endif;
A1:

    tstart = hsec;

    d = -h*g;

    { s,gflag,bksteps} = _stepl(g,vof,x0,d,&_mfct);
    if scalmiss(s);
        if trapchk(0);
            errorlog "step length calculation failed";
            end;
        else; 
            if __output == 1;
                cls;
            endif;
            retp(x0,vof,g,h);
        endif;
    endif;
    dx = s*d;
    x = x0 + dx;
    x0 = x;

    { g,h,vof} = _sctu(x,smallval,g,h,dx,&_mfct);

@  test for convergence  @

    if abs(g) < smallval;
        goto A98;
    endif;

    relgrad = abs(g).*(maxc(abs(x0)|1)/maxc(abs(vof)|1));
    relb = abs(dx)./maxc(abs(x0)|1);

    if abs(relb) < __btol;
        if abs(relgrad) < __gtol;
            goto A98;
        endif;
    endif;

    if __output == 1;
        gosub help;
    else;
        if key==1024;
            __output = 1;
        endif;
    endif;

    if iter >= __miter;
        goto A98;
    endif;

    dfct = f0-vof;
    iter = iter + 1;

    if __design;
        if dfct/f0 < __dfct or dsgn > __diter;
            dsgn = 0;
            __algr = 1;
            __step = 2;
            __design = 0;
            if __output == 1;
                scroll w1;
            endif;
            if __design == 1;
                h = _deriv(x0,2,&_mfct);
                if not scalmiss(h);
                    oldt = trapchk(1);
                    trap 1,1;
                    h = invpd(h);
                    trap oldt,1;
                endif;
                if scalmiss(h);
                    if __output == 1;
                        locate 2,1;
                        print "hessian failed to invert - H set to identity "\
                            "matrix";
                    endif;
                    h = eye(k)*(1/maxc(abs(vof)|1));
                endif;
            endif;
        else;
            dsgn = dsgn+1;
        endif;
    endif;

    goto A0;
A98:

    /* ******************** End of iteration loop ******************
    :: **** 
    */

    if maxl == 0;
        if __covp == 1;     /* covariance matrix of parameters */
            oldt = trapchk(1);
            trap 1,1;
            h = _deriv(x0,2,&_mfct);
            if not scalmiss(h);
                h = invpd(h);
            endif;
            if scalmiss(h);
                locate 2,1;
                errorlog "covariance matrix of parameters failed to invert";
                goto A99;
            endif;
            trap oldt,1;
        elseif __algr == 0;         /* if steepest descent and __covp=
                                    :: 0, 
                                    */
            h = { .};        /* then set covariance matrix to missing  */
        endif;
    endif;

A99:
    if __output == 1;
        cls;
    endif;
    call ndpcntrl(old,0xffff);
    format /rd 10,6;
    retp(x,vof,g,h);

HELP:

    ky = key;
    do while ky;
    A5:

        if ky eq 1048;      /* ALT B */
            __algr = 1;     /* BFGS algorithm */
        elseif ky eq 1032;          /* ALT D */
            __algr = 2;     /* DFP algorithm */
        elseif ky eq 1049;          /* ALT N */
            __algr = 3;     /* Newton-Raphson algorithm */
            __hsproc = __ml_hhs;
        elseif ky eq 1016 and maxl /= 0;    /* ALT Q */
            __algr = 4;     /* BHHH algorithm */
            __hsproc = __ml_scr;
        elseif ky eq 1031;          /* ALT S */
            __algr = 0;     /* steepest descent algorithm */
        elseif ky eq 1046;          /* ALT C */
            goto A98;       /* force convergence */
        elseif ky eq 1019;          /* ALT R */
            scroll w2;
            locate 8,4;
            "__COVP = ";;
            format 1,0;
            __covp;
            locate 10,4;
            " = 0, information matrix from final iteration";
            locate 11,4;
            " = 1, inverse of hessian";
            if maxl and __ml_scr;
                locate 12,4;
                " = 2, inverse of cross-product of first derivatives";
                locate 13,4;
                " = 3, heteroskedastic-consistent covariance matrix of para"\
                    "meters";
            endif;
            locate 15,4;
            "Enter new value - ";;
            call csrtype(1);
            k0 = cons;
            call csrtype(0);
            if k0 $/= "";
                __covp = stof(k0);
            endif;
            scroll w2;
            gosub parbox;
            locate 6,6;
            "parameters/gradient";
        elseif ky eq 1050;          /* ALT M */
            scroll w2;
            locate 8,4;
            "Maximum number of backsteps = ";;
            format 1,0;
            print __mxbkst;
            locate 10,4;
            "Enter new value - ";;
            call csrtype(1);
            k0 = cons;
            call csrtype(0);
            if k0 $/= "";
                __mxbkst = stof(k0);
            endif;
            scroll w2;
            gosub parbox;
            locate 6,6;
            print "parameters/gradient";
        elseif ky eq 1033;          /* ALT F */
            __grdmd = 1-__grdmd;
        elseif ky eq 1024;          /* ALT O */
            __output = 0;
            cls;
            print "output turned off.  Press Alt-O to reinstate.";
            goto A9;
        elseif ky eq 1022;          /* ALT U */
            scroll w2;
            locate 8,4;
            print "__BTOL coefficient convergence criterion = ";;
            print __btol;
            locate 10,4;
            print "Enter new value - ";;
            call csrtype(1);
            k0 = cons;
            call csrtype(0);
            if k0 $/= "";
                __btol = stof(k0);
            endif;
            scroll w2;
            gosub parbox;
            locate 6,6;
            print "parameters/gradient";
        elseif ky eq 1034;          /* ALT G */
            scroll w2;
            locate 8,4;
            print "__DFCT change in function criterion for __DESIGN = ";;
            print __dfct;
            locate 10,4;
            print "Enter new value - ";;
            call csrtype(1);
            k0 = cons;
            call csrtype(0);
            if k0 $/= "";
                __dfct = stof(k0);
            endif;
            scroll w2;
            gosub parbox;
            locate 6,6;
            print "parameters/gradient";
        elseif ky eq 1025;          /* ALT P */
            scroll w2;
            locate 8,4;
            print "__DITER no. of iters for change in algorithm = ";;
            print __diter;
            locate 10,4;
            print "Enter new value - ";;
            call csrtype(1);
            k0 = cons;
            call csrtype(0);
            if k0 $/= "";
                __diter = stof(k0);
            endif;
            scroll w2;
            gosub parbox;
            locate 6,6;
            print "parameters/gradient";
        elseif ky eq 1047;          /* ALT V */
            scroll w2;
            locate 8,4;
            print "__GTOL gradient convergence criterion = ";;
            print __gtol;
            locate 10,4;
            print "Enter new value - ";;
            call csrtype(1);
            k0 = cons;
            call csrtype(0);
            if k0 $/= "";
                __gtol = stof(k0);
            endif;
            scroll w2;
            gosub parbox;
            locate 6,6;
            print "parameters/gradient";
        elseif ky eq 1020;          /* ALT T */
            scroll w2;
            format 1,0;
            olddsgn = __design;
            locate 8,4;
            print "__DESIGN algorithmic strategy - current value ";;
            print __design;
            locate 9,4;
            print "  = 0, turn off strategy";
            locate 10,4;
            print "  = 1, invert Hessian upon switching to BFGS";
            locate 11,4;
            print "  = 2, set Hessian to identity";
            locate 13,4;
            print "Enter new value - ";;
            call csrtype(1);
            k0 = cons;
            call csrtype(0);
            if k0 $/= "";
                __design = stof(k0);
            endif;
            scroll w2;
            gosub parbox;
            locate 6,6;
            print "parameters/gradient";
            if __design and not olddsgn;
                dsgn = iter;
                __algr = 0;
                __step = 1;
            endif;
        elseif ky eq 1036;          /* ALT J */
            __step = 1;
        elseif ky eq 1037;          /* ALT K */
            __step = 2;     /* STEPBT step length */
        elseif ky eq 1038;          /* ALT L */
            __step = 3;     /* Golden step length */

        elseif ky eq 1018;          /* ALT E */
            scroll 6|1|6|80|0|7;
            locate 5,2;
            print "EDIT PARAMETER VECTOR";;
            print "    <> <> to move   <ENTER> to select   <Q> to quit";;
            k1 = 1;
        A3:

            locate 6,4;
            format /ldn 3,0;
            print "par. no. ";;
            print k1;;
            print "  old value ";;
            format 9,5;
            print x0[k1];;
            ky = key;
            do while ky;
                if ky eq 1072;
                    k1 = maxc(1|k1-1);
                elseif ky eq 1080;
                    k1 = minc(rows(x0)|k1+1);
                elseif ky eq 13;
                    locate 6,40;
                    print "new value ";;
                    call csrtype(1);
                    x0[k1] = con(1,1);
                    call csrtype(0);
                    scroll 6|37|6|80|0|7;
                    goto A3;
                elseif ky eq 81 or ky eq 113;
                    goto A4;
                endif;
                ky = key;
            endo;
            goto A3;
        A4:

            scroll w2;
            x = x0;
            gosub parbox;
            locate 6,6;
            print "parameters/gradient";

            scroll w1;
            if __algr <= 2;
                locate 2,1;
                print "H set to identity matrix";
                if __algr eq 0;
                    h = 1;
                else;
                    h = eye(k)*(1/maxc(abs(vof)|1));
                endif;

            elseif __algr gt 3;
                oldt = trapchk(1);
                trap 1,1;
                h = _deriv(x0,2,&_mfct);
                if not scalmiss(h);
                    h = invpd(h);
                endif;
                if scalmiss(h);
                    locate 2,1;
                    print "hessian matrix failed to invert - algorithm"\
                          " switched to BFGS";
                    __algr = 1;
                    h = eye(k)*(1/maxc(abs(vof)|1));
                endif;
                trap oldt,1;
            endif;
            scroll w1;

        elseif ky eq 1023;          /* ALT I */
            scroll w1;
            if __algr ne 0;
                h = _deriv(x0,2,&_mfct);
                if not scalmiss(h);
                    oldt = trapchk(1);
                    trap 1,1;
                    h = invpd(h);
                    trap oldt,1;
                endif;
                if scalmiss(h);
                    locate 2,1;
                    print "hessian failed to invert - H set to identity"\
                          " matrix";
                    h = eye(k)*(1/maxc(abs(vof)|1));
                endif;
            endif;
        elseif ky eq 1035;          /* ALT H */
            scroll 6|1|24|80|0|7;
            locate 7,4;
            print "OPTIMIZATION SWITCHES";
            k0 = 196*ones(1,30);
            locate 7,4;
            print chrs(218~k0~194~k0~191);
            k4 = 8;
            if __design;
                k4 = 10;
            endif;
            if __ml_scr;
                k4 = maxc(k4|9);
            endif;
            k1 = 0;
            do until k1 == k4;
                k1 = k1+1;
                locate 7+k1,4;
                print chrs(179~zeros(1,30)~179~zeros(1,30)~179);
            endo;
            locate 8+k1,4;
            print chrs(192~k0~193~k0~217);
            locate 8,6;
            print "ALT J  Step Length = 1       ";
            locate 9,6;
            print "ALT K  Quad. & Cubic Step    ";
            locate 10,6;
            print "ALT L  Golden Step length    ";
            locate 11,6;
            print "ALT F  Toggle __GRDMD        ";
            locate 12,6;
            print "ALT U  __BTOL                ";
            locate 13,6;
            print "ALT V  __GTOL                ";
            locate 14,6;
            print "ALT R  __COVP                ";
            locate 15,6;
            print "ALT T  Toggle __DESIGN       ";
            if __design;
                locate 16,6;
                print "ALT G  __DFCT                ";
                locate 17,6;
                print "ALT P  __DITER               ";
                locate 17,37;
            else;
                locate 16,6;
            endif;
                print "ALT O  __OUTPUT = 0          ";

            locate 8,37;
            print "ALT M  Maximum Backstep      ";
            locate 9,37;
            print "ALT I  Compute Hessian       ";
            locate 10,37;
            print "ALT E  Edit Parameter Vector ";
            locate 11,37;
            print "ALT C  Force Convergence     ";
            locate 12,37;
            print "ALT S  Steepest Descent      ";
            locate 13,37;
            print "ALT B  BFGS Algorithm        ";
            locate 14,37;
            print "ALT D  DFP Algorithm         ";
            locate 15,37;
            print "ALT N  Newton Algorithm      ";
            if maxl;
                locate 16,37;
                print "ALT Q  BHHH Algorithm        ";
            endif;

            ky = key;
            do until ky;
                ky = key;
            endo;
            if ky /= 1019 and ky /= 1022 and ky /= 1047 and ky /= 1050 and 
                ky /= 1020 and ky /= 1034 and ky /= 1025;
                gosub parbox;
            endif;
            goto A5;
        elseif ky == 1073 or ky == 1081;    /* PgUp PgDn */
            if ky eq 1081 and (k-pg*48) gt 0;
                pg = pg+1;
            elseif ky eq 1073 and pg gt 1;
                pg = pg-1;
            else;
                goto A9;
            endif;
        endif;
    A9:

        ky = key;
    endo;
    return;

BAR:

    scroll w3;
    Scroll w4;
    printdos "\27[7m";
    locate 1,1;
    print w0;
    locate 3,2;
    print "ITER      ";;
    locate 3,19;
    print "FUNCTION:";;
    locate 4,2;
    print "TIME/ITER:";
    locate 3,41;
    print "ALGORITHM:";;
    locate 3,60;
    print "STEP:";;
    locate 4,60;
    print "STEPSIZE:";;
    locate 3,52;
    format /ld 6,6;
    print $algrm[__algr+1];;
    locate 3,66;
    print $stepm[__step];;
    locate 4,19;
    print "DF/ITER :";;
    locate 4,41;
    print "BACKSTEPS:";;
    locate 25,1;
    print "  ALT-H HELP  ";;
    printdos "\27[0m";
    return;

PARBOX:

    scroll w5;
    locate 6,6;
    print "parameters/relative gradient";;
    lf = (pg-1)*48+1;
    ll = minc(48|(k-(pg-1)*48));
    lr = ceil(ll/3);
    locate 7,1;
    print w6;
    k1 = 1;
    do until k1 gt lr;
        locate 7+k1,1;
        print w7;
        k1 = k1+1;
    endo;
    print w8;
    return;
endp;

@-----------------------------------------------------@
@   PROC SCTU                                         @
@ This computes VOF, G & updates to inverse Hessian   @

proc(3) = _sctu(x,smallval,g,h,dx,&_mfct);

@------- LOCALS --------@
    local shy, shys, yshy, ys, v, g0, h1, vof, oldt;

@------ EXTERNALS -------@
    external proc _deriv;
    local gdproc:proc, hsproc:proc, _mfct:proc;

    h1 = h;
    vof = _mfct(x);
    if scalmiss(vof);
        retp(g,h1,error(0));
    endif;
    /* --- Gradient at x --- */
    g0 = g;
    g = _deriv(x,1,&_mfct)';        /* Gradient vector. */

    if abs(g) < smallval;
        goto endscmm;
    endif;

    /* -- Secant Update for Inverse Hessian --- */

    if __algr eq 1;         /* BFGS update. */
        shy = dx-h*g+h*g0;
        shys = shy*dx';
        yshy = g'*shy-g0'*shy;
        ys = g'*dx-g0'*dx;
        h1 = h + ((shys + shys') - (yshy*(dx*dx'))/ys)/ys;

    elseif __algr eq 2;     /* DFP update. */
        h1 = h+dx*dx'/(dx'*g-dx'*g0);
        v = h*g-h*g0;
        h1 = h1-v*v'/(g'*v-g0'*v);

    elseif __algr ge 3;
        oldt = trapchk(1);
        trap 1,1;
        h1 = _deriv(x,2,&_mfct);
        if not scalmiss(h1);
            h1 = invpd(h1);
        endif;
        if scalmiss(h1);
            __algr = 1;
            if __output == 1;
                locate 2,1;
            endif;
            errorlog "Hessian matrix failed to invert - "\
                     "algorithm switched to BFGS";
            h1 = eye(rows(x))*(1/maxc(abs(vof)|1));
        endif;
        trap oldt,1;
    else;
        h1 = 1/maxc(abs(vof)|1);
    endif;

endscmm:

    retp(g,h1,vof);
endp;

proc(3) = _stepl(g,vof,x0,d,&_mfct);
    local s, rs, ret, goldeps, gflag, bksteps, _mfct:proc;
    external proc __stepbt, _golden;

    clear gflag,ret;
    bksteps = -1;
    goldeps = 1e-2/abs(vof);        /* tolerance used for Golden search
                                    :: algorithm 
                                    */

    if __step == 2;
        s = 1;
        rs = _mfct(x0+s.*d);
        if scalmiss(rs);
            retp(error(0),gflag,bksteps);
        endif;
        { s,ret,rs,bksteps} = __stepbt(rs,g,vof,x0,d,&_mfct);
        if ret==1;          /* not successful */
            { s,ret,rs} = _golden(1.5,goldeps,x0,d,&_mfct);
            gflag = 1;
            if scalmiss(s);
                retp(s,gflag,bksteps);
            endif;
        endif;
    elseif __step == 3;
        { s,ret,rs} = _golden(1.0,goldeps,x0,d,&_mfct);
        if scalmiss(s);
            retp(s,gflag,bksteps);
        endif;
        bksteps = -1;
    else;
        s = 1;
        rs = _mfct(x0+s'.*d);
        if scalmiss(rs);
            retp(error(0),gflag,bksteps);
        endif;
        if rs > vof;        /* jump to STEPBT */
            { s,ret,rs,bksteps} = __stepbt(rs,g,vof,x0,d,&_mfct);
            if ret==1;      /* not successful */
                { s,ret,rs} = _golden(1.5,goldeps,x0,d,&_mfct);
                gflag = 1;
                if scalmiss(s);
                    retp(s,gflag,bksteps);
                endif;
            endif;
        endif;

    endif;
    if ret eq 1;
        retp(error(0),error(0),error(0));
    endif;

    retp(s,gflag,bksteps);
endp;

proc(4) = __stepbt(r1,g,vof,x0,d,&_mfct);
    local delta,ub,lb, ret, i, cdelta, dg, s, g1, r2, rs, sprev, s2prev, 
        tt, rprev, r2prev, sprev2, s2prev2, sp2, dsprev, vv, zz, ab, a, b, 
        rr0, rr1, qv;

    local _mfct:proc;

/* --------------------- Initializations -------------------------  */
    delta = 1e-4;           /* This can be changed, and doing so may help
                            :: speed 
                            */
                /* convergence -- it must remain within the interval  */
            /* (0,1/2) */
    ub = 0.5;       /* Upper bound on acceptable reduction in s. */
    lb = 0.1;       /* Lower bound on acceptable reduction in s. */

    ret = 1;        /* If 0, then satisfactory value found; else 1.  */
    i = 0;          /* This counts # of backsteps taken. */

    cdelta = 1-delta;

    dg = d'*g;

        /* ------------------- Try s=1 -------------------------- */
    s = 1;
    tt = s*dg;
    g1 = r1/tt-vof/tt;
    if g1 ge delta;
        rs = r1;
        ret = 0;
        goto termnate;
    endif;
    i = 1;
    s = -dg/(2*(r1-vof-dg));
    s = maxc(s|lb);
    r2 = _mfct(x0+s'.*d);
    if scalmiss(r2);
        retp(error(0),1,r2,i);
    endif;
    tt = s*dg;
    g1 = r2/tt-vof/tt;

    if g1 ge delta and g1 le cdelta;
        rs = r2;
        ret = 0;
        goto termnate;
    endif;
    sprev = s;
    s2prev = 1;
    rprev = r2;
    r2prev = r1;

    i = 2;
    do until i > __mxbkst;

        sprev2 = sprev*sprev;
        s2prev2 = s2prev*s2prev;
        sp2 = sprev2~s2prev2;
        dsprev = sprev-s2prev;

        vv = (1~-1|-s2prev~sprev);
        vv = vv./sp2;
        zz = (rprev-vof-dg*sprev)|(r2prev-vof-dg*s2prev);
        ab = (1/dsprev)*vv*zz;
        a = ab[1,1];
        b = ab[2,1];

        if a == 0;          /* Cubic is actually a Quadratic in this case. */
            s = -dg/(2*b);
        else;
            qv = b*b - 3*a*dg;
            if qv < 0;
                break;
            endif;          /* terminate if not real root */
            tt = 3*a;
            s = -b/tt + sqrt(qv)/tt;
        endif;

        if s > ub*sprev;
            s = ub*sprev;
        elseif s < lb*sprev;
            s = lb*sprev;
        endif;

        rs = _mfct(x0+s'.*d);
        if scalmiss(r2);
            retp(error(0),1,rs,i);
        endif;
        tt = s*dg;
        g1 = rs/tt-vof/tt;

        if g1 ge delta and g1 le cdelta;
            ret = 0;
            break;
        endif;

        s2prev = sprev;
        sprev = s;
        r2prev = rprev;
        rprev = rs;
        i = i+1;
    endo;
termnate:

    retp(s,ret,rs,i);
endp;

proc (3) = _golden(t,eps,x0,d,&_mfct);
    local maxtries, ret, v1, v2, cd, cs, vr0, vr1, vr2, a0, b0, g0, g1, g2,
        l, i, rd, rs, del, sig, s;
    local _mfct:proc;

    /* --------------- initializations ------------------- */
    maxtries = 100;         /* Will stop trying to find min after this many
                            :: tries 
                            */
    ret = 1;        /* If 0, then search was successful; otherwise 1  */
    v1 = (3-sqrt(5))/2;
    v2 = (sqrt(5)-1)/2;
    cd = 1;
    cs = 1;         /* if 1, compute rd, rs, respectively, in seven below  */

/* --------------- test to see if [0,t] brackets minimum ---------------- */
    vr0 = _mfct(x0);
    if scalmiss(vr0);
        retp(error(0),1,vr0);
    endif;
    vr1 = _mfct(x0+t'.*d);
    if scalmiss(vr1);
        retp(error(0),1,vr1);
    endif;
    if vr0 <= vr1;
        a0 = 0;
        b0 = t;
        goto seven;
    endif;

    g0 = 0;
    g1 = t;

/* -- if not, expand interval by increments of t until min is bracketed -- */
four:

    g2 = g1+t;
    vr2 = _mfct(x0+g2'.*d);
    if scalmiss(vr2);
        retp(error(0),1,vr2);
    endif;
    if vr2 >= vr1;
        a0 = g0;
        b0 = g2;
        goto seven;
    else;
        vr1 = vr2;
        g0 = g1;
        g1 = g2;
        goto four;
    endif;

/* ---------------- search to isolate minimum ---------------------------- */
seven:

    i = 1;
    do until i > maxtries;
        l = b0-a0;
        if l < eps;
            ret = 0;
            goto fin;
        endif;

        if cd;
            del = a0+v1*l;
            rd = _mfct(x0+del'.*d);
            if scalmiss(rd);
                retp(error(0),1,rd);
            endif;
        endif;
        if cs;
            sig = a0+v2*l;
            rs = _mfct(x0+sig'.*d);
            if scalmiss(rs);
                retp(error(0),1,rs);
            endif;
        endif;

        if rd < rs;
            b0 = sig;
            rs = rd;
            sig = del;
            cs = 0;
            cd = 1;
        else;
            a0 = del;
            rd = rs;
            del = sig;
            cs = 1;
            cd = 0;
        endif;

        i = i+1;
    endo;

fin:

    s = (a0+b0)/2;
    rs = _mfct(x0+s'.*d);
    retp(s,ret,rs);
endp;

proc _deriv(x,ind,&_mfct);
    local _mfct:proc;
    local temp,a,y,a0,k0,k1,k2,k3,y,i,nc;
    local hsproc,gdproc;

    if __hsproc /= 0;
        hsproc = __hsproc;
        local hsproc:proc;
    endif;

    if __gdproc /= 0;
        gdproc = __gdproc;
        local gdproc:proc;
    endif;

        if ind == 1;
           if __output == 1;
               locate 2,15;
               print "gradient";
           endif;
        elseif ind == 2;
            if __algr == 4;
              if __output == 1;
                  locate 2,15;
                  print "information matrix";
              endif;
            else;
              if __output == 1;
                  locate 2,15;
                  print "hessian";
              endif;
            endif;
        else;
            if trapchk(0);
                errorlog "ERROR: incorrect argument _DERIV";
                end;
            else;
                retp(error(41));
            endif;
        endif;

    if ind == 1;
        if __gdproc /= 0;
            a = gdproc(x);
        else;
            temp = _grad_dh;
            _grad_dh = __dh;
            if __grdmd;
                a = gradfd(&_mfct,x);
            else;
                a = gradcd(&_mfct,x);
            endif;
            _grad_dh = temp;
        endif;
    elseif ind == 2;
        if __hsproc /= 0;
            a = hsproc(x);
        else;
            a = hessp(&_mfct,x);
        endif;
    endif;
    if scalmiss(a) and trapchk(0);
        errorlog "Calculation of gradient failed";
        end;
    endif;
    if __output == 1;
        let i = 2 15 2 25 0 7;
        scroll i;
    endif;
    retp(a);
endp;

proc(0) = optset;

    __title = "";

    __algr = 1;     /* optimization algorithm */
    __covp = 1;     /* selects covariance matrix of parameters */
    __parnms = 0;           /* parameter names */
    __step = 2;     /* selects type of step length */
    __hess0 = 0;    /* selects starting hessian */
    __mxbkst = 10;          /* number of backsteps in computing steplength  */
    __btol = 1e-4;          /* convergence tolerance for coefficients  */
    __gtol = 1e-5;          /* convergence tolerance for gradient */
    __hsproc = 0;           /* procedure to compute hessian */
    __gdproc = 0;           /* procedure to compute gradient */
    __grdmd = 1;    /* numerical gradient method */
    __dh = 0;       /* size of increment for computing gradient */
    __ivcp = 0;     /* info matrix v-c matrix of par's */
    __cpvcp = 0;    /* cross proc matrix v-c matrix of par's */
    __design = 0;           /* combination of algorithms */
    __diter = 20;           /* # of iters to switch algorithms for _
                            :: _design 
                            */
    __dfct = .001;          /* % change in function for __design */
    __miter = 1e+5;         /* maximum number of iterations */
    __output = 1;

endp;

proc(4) = optprt(x,f,g,h);
    local lbl,mask,fmt,d;

    print;
    print;
    format /ro 15,6;
    print "value of objective function " f;
    print;
    print "parameters    estimates    gradient";
    print "-----------------------------------------";

    if __parnms == 0;
        lbl = 0 $+ "P" $+ ftocv(seqa(1,1,rows(x)),2,0);
    else;
        lbl = __parnms;
    endif;
    mask = 0~1~1;
    let fmt[3,3] = "-*.*s" 9 8 "*.*lf" 14 6 "*.*lf" 14 6;
    d = printfm(lbl~x~g,mask,fmt);

    print;
    print;
    format /ro 15,6;
    print "inverse of the hessian matrix";
    print;
    print h;
    retp(x,f,g,h);
endp;
