function [x,F] = nSecant( Fname,x0,noprnt,points )

global alpha dfactor nu delta rho sigma po_k po_t tmin tmax kmin kmax;
global maxiter psitol;
%
%	November 9 1998
%
%  	 [x,F,J] = nSecant( Fname,x0 ) finds  x  such that 
%        F(x) = 0 for a vector of nonlinear functions  F(x). 
%        The zero is found using the (n+1)-point secant method.
%
%        Fname is a string containing the name of an m-file that
%        computes the vector F.  x0 is an initial guess at the 
%        solution.  For example, secant( 'my_F',x0 ) finds a 
%        zero of the nonlinear function defined in the mfile
%        my_F.m.  Note the quotes around 'my_F'.
%
%        On successful termination, x is the first point found at 
%        which the norm of F is less than sqrt(eps).  Output 
%        argument is   F(x) at the final x.
%
%        The meaning of each line of printed output is as follows
%
%        Itn          The current iteration number.
%
%        Norm F       The current value of the norm of F(x). 
%                     In the neighborhood of a zero of F, 
%                     Norm F will be small.
%
%        This version dated 07-Oct-1995.
%        Philip E. Gill, University of California, San Diego.
%
%        This version dated 01-May-1998.
%	 Modified by Christian Haefke to suit the problem
%        Christian Haefke, University of California, San Diego.
% -------------------------------------------------------------------
[n,m]     = size(x0);

itn_limit = maxiter;		% Limit on evaluations of F.
tol       = psitol;     	% Required accuracy in norm F.

h         = sqrt(eps); 		% Rule-of-thumb optimal h.
k         = 0;
x         = x0;                     
[F]       = feval( Fname,x,points );
normF     = norm( F );

% Start with the finite-difference Jacobian.

j         = 0;
for    e  = eye(n),
   xd     = x + h*e;
   [Fd]   = feval( Fname,xd,points );
   j      = j + 1;
   Y(:,j) = Fd - F;
end

S         = h*eye(n);
P         = 1:n;
P         = P + ones(size(P));
P(n)      = 1;
 
if noprnt == 0
%   clc                                 
   fprintf( ' Itn      Norm F\n' )
   str    = sprintf ( ' %3g    %14.7e', k, normF ); 
   disp( [ str ] );
end

while  normF >= tol  &  k < itn_limit

   p       = - S*(Y\F);
   xnew    = x + p;
   %xnew    = x + p;
   [Fnew]  = feval( Fname,xnew,points );

   S(:,n)  = xnew - x;                S = S(:,P);
   Y(:,n)  = Fnew - F;                Y = Y(:,P);

   x       = xnew;
   F       = Fnew;

   k       = k + 1;

   normF   = norm( F );
   if noprnt == 0
      str    = sprintf ( ' %3g    %14.7e', k, normF ); 
      disp( [ str ] );
   end
end

if     normF <= tol  
   status = 'Solved    ';           
else  
   status = 'k too big ';
end

if     status == 'Solved    '
   fprintf( '\nSolved at iteration %g.  ', k )
   fprintf( '  Norm f = %14.7e\n\n', normF );
elseif status == 'k too big '
   fprintf( '\nSecant terminated.  Iteration limit exceeded    \n\n' )
end

% **********************************************************************

% **********************************************************************
