% KKK_simul.m
% Master file to calculate main statistics from simulation
% Simulation is based on given individual and aggregate shocks for 10,000 periods and initial distribution of capital. 
% separating employed and unemployed for bad and good states. 

clear all;
%---------------------------------------------------------------
% define Global variables
%---------------------------------------------------------------

global NC BETA  ALPHA GAM AADDJJ DEPREC REPLACEMENT LabSS KSS LOGQ ORD
global FD FDD M11 M2 zs zu sig pick C six usix omega AI_SHOCKS
global xss1 xss3 ACC delo indKe indKu aggK

%---------------------------------------------------------------
% Select options in simulations
%---------------------------------------------------------------

ACC=0;  % 1 only if calculating approximation errors (use absolute values instead of actual values for budget constraint), otherwise 0
AI_SHOCKS=2;  % 1 for individual shocks only and 2 for aggregate AND individual shocks  
LOGQ=0; % 0 for level and 1 for logs
ORD=1; % 1 for linear and 2 for quadratic: order of approximation
NC=1;  % 1 if we use approximate aggregation property and treat the rest of agents as representative agent
delo=[0 0];  % should be always zero. no alternate adjustment

if ((AI_SHOCKS-1)*(AI_SHOCKS-2))~=0; error('____2'); end;
if ((ORD-1)*(ORD-2))~=0; error('____2'); end;
if ((LOGQ)*(LOGQ-1))~=0; error('____'); end;

%---------------------------------------------------------------
% define parameter values
%---------------------------------------------------------------

AADDJJ=0.03;  %parameter value fo the coefficient in the barrier function
BETA=.99;
ALPHA=.36;
GAM=1; 
DEPREC=.025;
if AI_SHOCKS==1; LabSS=.90;  end; % SS employment
if AI_SHOCKS==2; LabSS=.93;  end;
REPLACEMENT=.15;

if AI_SHOCKS==1;   %only individual shocks
omega=[1 0;0 1e-10]; % shock variance (converted to a continuous time version), normalized to 1
end;

if AI_SHOCKS==2;  % with noth individual and aggregate shocks
omega=eye(2);
end;

sig=1;  % scale parameter for exogenous shocks, always 1 in thie simulation

%---------------------------------------------------------------
% Steady state
%---------------------------------------------------------------

KSS=(LabSS/.9)*((DEPREC+(1-BETA)/BETA)/ALPHA)^(1/(ALPHA-1));
CSS=(KSS^ALPHA)*((LabSS/.9)^(1-ALPHA))-DEPREC*KSS;

%---------------------------------------------------------------
% Define perturbation variables
%---------------------------------------------------------------

if LOGQ==0;  % perturbation variables are in levels
xss1=[CSS; LabSS; KSS; CSS; LabSS; KSS; 1];    
end;

if LOGQ==1;  % perturbation variables are in logs
xss1=[log(CSS); LabSS; log(KSS); log(CSS); LabSS; log(KSS); 1];    
end;


xss3=zeros(2,1);
xss=[xss1;xss1;xss3];  % variables at time t , t-1 and shocks at steady states

%---------------------------------------------------------------
% calculating inputs for gensys2
%---------------------------------------------------------------

c=jedcB1f(xss);    % Use the jedcB1f function, solving the model, c is residuals from the SS. 
if maxa(c)>1e-7; error('');  end;
if rows(c)~=7; error('ABCABC'); end;

Pi=zeros(rows(c),2); %matrix denoting where the shocks are
Pi(1:2,:)=eye(2);

pick=zeros(4,rows(xss1));   % matrix denoting where the expectational error terms are.
pick(:,[2 3 6 7])=eye(4);

[KD,KDD,KDDALL]=pregen('jedcB1f',xss1,xss3); %calculating numerical derivatives (first and second order) from the model

%---------------------------------------------------------------
% solve the model using gensys2
%---------------------------------------------------------------


[FD,FDD,M11,M2,C,q,zs,zu,v,gev,eu,six,usix]=gensys2(KD,KDD,c*0,Pi,omega,pick);  % using gensys2 to solve the model

if isempty(FD)==1; error('FD is empty'); end;

if maxaa(imag(FD{1}))>1e-8; error('FD{1} imaginary'); end;
if maxaa(imag(FD{2}))>1e-8; error('FD{2} imaginary'); end;
if maxaa(imag(FD{3}))>1e-8; error('FD{3} imaginary'); end;

% ----------------------------------------------------------------------
% now load the given individual and aggregate shocks for 10,000 periods.
% ----------------------------------------------------------------------

load ind_switch.txt  %2 means employed, 1 means unemployed
indsh=ind_switch-ones(size(ind_switch));  % now 1 means employed and 0 means unemployed
load agg_switch.txt  %1 means bad state and 2 means good state
%agg_switch=[ones(100,1);ones(100,1)*2];  %100 periods of bad state and then 100 periods of good state for another set of simulations
aggsh=ones(size(agg_switch))*.99+(agg_switch-ones(size(agg_switch)))*.02;  % now .99 means bad state and 1.01 means good state

aggsh=[aggsh;.99];  % in order to calculate the 10000th period, give ransom shock for 10,001th period (results do not change)

load empl.txt;  % initial distribution of capital for employed agents (taken from pdist.xls)
load unempl.txt;  % initial distribution of capital for unemployed agents (taken from pdist.xls)
empl=empl/sum(empl);  %make sure that the integral of probability density is normalized to one
unempl=unempl/sum(unempl);

disp('all shocks are loaded')


% ------------------------------------------------
% Calculating initial period statistics
% ------------------------------------------------

STAT=[];
statis=cross_RK(empl, unempl, aggsh(1,:)); STAT=[STAT;statis];  %use the proc cross_RK.m to calculate cross-sectional statstics

%implement the nonstochastic simulation to calculate next period's individual and aggregate capital (given agg shocks and capital distributions)
[indKe, indKu, aggKu, aggKe, aggK]=aggcapital_RK(empl, unempl, aggsh(1:2,:));  

%from the given capital dist, calculate optimal policy functions using jedcB1_POLF_pos.m
wL=jedcB1_POLF_pos([indsh(1,:)-LabSS;43-KSS;aggK-KSS;aggsh(1,:)-1],delo); %given individual capital at 43
hL=(wL+xss1)'; hL(3)=43; hL(6)=aggK; % note that we record time t ki and K for output

wage=(1-ALPHA)*hL(7).*(aggK./(hL(5)/.9)).^ALPHA;    
rental=ALPHA*hL(7).*(aggK./(hL(5)/.9)).^(ALPHA-1); 

res=[agg_switch(1,:) ind_switch(1,:) aggK 43 wL(1,:)+CSS aggKu aggKe rental wage];  %variables to be saved

% ------------------------------------------------
% Calculating statistics from the second period and on
% ------------------------------------------------

j=2; while j<rows(aggsh);
j  %period

indKL=wL(3,:);
statis=cross_RK(indKe, indKu, aggsh(j,:)); STAT=[STAT; statis];  %calculating cross-sectional statistics in each period

[indKe, indKu, aggKu, aggKe, aggK]=aggcapital_RK(indKe, indKu, aggsh(j:j+1,:));  

wL=jedcB1_POLF_pos([indsh(j,:)-LabSS;indKL;aggK-KSS;aggsh(j,:)-1],delo); 
hLL=(wL+xss1)'; hLL(3)=indKL+KSS; hLL(6)=aggK; 
hL=[hL;hLL];

wage=(1-ALPHA)*hLL(7).*(aggK./(hLL(5)/.9)).^ALPHA;    
rental=ALPHA*hLL(7).*(aggK./(hLL(5)/.9)).^(ALPHA-1); 

res=[res; [agg_switch(j,:) ind_switch(j,:) aggK indKL+KSS wL(1,:)+CSS aggKu aggKe rental wage]];  %variables to be saved

j=j+1; end;
% ..................................................................

%---------------------------------------------------------------------
% Saving the results into two mat files
%---------------------------------------------------------------------

save res res -ASCII  %save 7 variables for 10,000 periods from simulation
save hL hL ALPHA REPLACEMENT KSS LabSS DEPREC BETA GAM ;
save STAT STAT -ASCII  %cross-sectional statistics

disp('calculated statistics from non-stochastic simulations')

%return


% FOR DEN HAAN ACCUEACY TEST and COMPARISON, SECT. 8.2 and 8.3
% ADDED SIMULATION USIND APPROXIMATE AGGREGATE LAW OF MOTION (for  K(t))

% STAT(:,end-1:end): mean capital of unemployed and employed with ki>0, 
%  based on aggregate capital obtained by NON-STOCHASTIC simulation procedure 

% STAT_AG(:,end-1:end): mean capital of unemployed and employed with ki>0
%  based on first-order law of motion of aggregate capital

% initial period

STAT_AG=[];
stats_AG=cross_RK(empl, unempl, aggsh(1,:)); STAT_AG=[STAT_AG; stats_AG];

[indKe_AG, indKu_AG, aggKu_AG, aggKe_AG, aggK_AG]=aggcapital_RK(empl,unempl,aggsh(1:2,:));

wL_AG=jedcB1_POLF_pos([indsh(1,:)-LabSS;43-KSS;aggK_AG-KSS;aggsh(1,:)-1],delo); 
hL_AG=(wL_AG+xss1)'; hL_AG(3)=43; hL_AG(6)=aggK_AG;
%hL=[ci(t),ei(t),ki(t+1),C(t),L(t),K(t),theta(t)

% !!!!!!!!!!!!!!!!
K_ALM=wL_AG(6,:)+KSS; % aggregate K(2) based on ALM (aggregate law of motion)
% !!!!!!!!!!!!!!!!

% from subsequent peirods

j=2; while j<rows(aggsh);
%j=2; while j<10;
j
indKL_AG=wL_AG(3,:);

stats_AG=cross_RK(indKe_AG, indKu_AG, aggsh(j,:)); STAT_AG=[STAT_AG; stats_AG];
[indKe_AG, indKu_AG, aggK_AG]=aggcapital_ALM(indKe_AG,indKu_AG,K_ALM,aggsh(j:j+1,:));  % now use aggcapital_ALM to reflect using aggregate law of motion to calculate next period's capital

wL_AG=jedcB1_POLF_pos([indsh(j,:)-LabSS;indKL_AG;K_ALM-KSS;aggsh(j,:)-1],delo);
hLL_AG=(wL_AG+xss1)'; hLL_AG(3)=indKL_AG+KSS; hLL_AG(6)=K_ALM; hL_AG=[hL_AG;hLL_AG]; % hL: K(t) from ALM 

K_ALM=wL_AG(6,:)+KSS; % aggregate K(j+1) based on ALM (aggregate law of motion)

j=j+1; end;
% ..................................................................

save hL_AG hL_AG ALPHA REPLACEMENT KSS LabSS DEPREC BETA GAM ;
save STAT_AG STAT_AG -ASCII
